/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Below are constants for LView indices to help us look up LView members
// without having to remember the specific indices.
// Uglify will inline these when minifying so there shouldn't be a cost.
export const HOST = 0;
export const TVIEW = 1;
export const FLAGS = 2;
export const PARENT = 3;
export const NEXT = 4;
export const TRANSPLANTED_VIEWS_TO_REFRESH = 5;
export const T_HOST = 6;
export const CLEANUP = 7;
export const CONTEXT = 8;
export const INJECTOR = 9;
export const RENDERER_FACTORY = 10;
export const RENDERER = 11;
export const SANITIZER = 12;
export const CHILD_HEAD = 13;
export const CHILD_TAIL = 14;
// FIXME(misko): Investigate if the three declarations aren't all same thing.
export const DECLARATION_VIEW = 15;
export const DECLARATION_COMPONENT_VIEW = 16;
export const DECLARATION_LCONTAINER = 17;
export const PREORDER_HOOK_FLAGS = 18;
export const QUERIES = 19;
export const ID = 20;
export const EMBEDDED_VIEW_INJECTOR = 21;
/**
 * Size of LView's header. Necessary to adjust for it when setting slots.
 *
 * IMPORTANT: `HEADER_OFFSET` should only be referred to the in the `ɵɵ*` instructions to translate
 * instruction index into `LView` index. All other indexes should be in the `LView` index space and
 * there should be no need to refer to `HEADER_OFFSET` anywhere else.
 */
export const HEADER_OFFSET = 22;
/**
 * Converts `TViewType` into human readable text.
 * Make sure this matches with `TViewType`
 */
export const TViewTypeAsString = [
    'Root',
    'Component',
    'Embedded', // 2
];
// Note: This hack is necessary so we don't erroneously get a circular dependency
// failure based on types.
export const unusedValueExportToPlacateAjd = 1;
//# sourceMappingURL=data:application/json;base64,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