/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
/** Counter used to generate unique IDs for component definitions. */
let componentDefCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const type = componentDefinition.type;
        const standalone = componentDefinition.standalone === true;
        const declaredInputs = {};
        const def = {
            type: type,
            providersResolver: null,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            factory: null,
            template: componentDefinition.template || null,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            hostBindings: componentDefinition.hostBindings || null,
            hostVars: componentDefinition.hostVars || 0,
            hostAttrs: componentDefinition.hostAttrs || null,
            contentQueries: componentDefinition.contentQueries || null,
            declaredInputs: declaredInputs,
            inputs: null,
            outputs: null,
            exportAs: componentDefinition.exportAs || null,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            standalone,
            dependencies: standalone && componentDefinition.dependencies || null,
            getStandaloneInjector: null,
            selectors: componentDefinition.selectors || EMPTY_ARRAY,
            viewQuery: componentDefinition.viewQuery || null,
            features: componentDefinition.features || null,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: `c${componentDefCount++}`,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            setInput: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        const dependencies = componentDefinition.dependencies;
        const feature = componentDefinition.features;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach((fn) => fn(def));
        def.directiveDefs = dependencies ?
            (() => (typeof dependencies === 'function' ? dependencies() : dependencies)
                .map(extractDirectiveDef)
                .filter(nonNull)) :
            null;
        def.pipeDefs = dependencies ?
            (() => (typeof dependencies === 'function' ? dependencies() : dependencies)
                .map(getPipeDef)
                .filter(nonNull)) :
            null;
        return def;
    });
}
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = () => (typeof directives === 'function' ? directives() : directives).map(extractDirectiveDef);
    def.pipeDefs = () => (typeof pipes === 'function' ? pipes() : pipes).map(getPipeDef);
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
export const autoRegisterModuleById = {};
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        if (def.id != null) {
            autoRegisterModuleById[def.id] = def.type;
        }
        return res;
    });
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export const ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
//# sourceMappingURL=data:application/json;base64,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