/// <amd-module name="@angular/compiler-cli/ngcc/src/ngcc_options" />
import { AbsoluteFsPath, FileSystem } from '../../src/ngtsc/file_system';
import { Logger } from '../../src/ngtsc/logging';
import { ParsedConfiguration } from '../../src/perform_compile';
import { PathMappings } from './path_mappings';
import { FileWriter } from './writing/file_writer';
import { PackageJsonUpdater } from './writing/package_json_updater';
/**
 * The options to configure the ngcc compiler for synchronous execution.
 */
export interface SyncNgccOptions {
    /** The absolute path to the `node_modules` folder that contains the packages to process. */
    basePath: string;
    /**
     * The path to the primary package to be processed. If not absolute then it must be relative to
     * `basePath`.
     *
     * All its dependencies will need to be processed too.
     *
     * If this property is provided then `errorOnFailedEntryPoint` is forced to true.
     */
    targetEntryPointPath?: string;
    /**
     * Which entry-point properties in the package.json to consider when processing an entry-point.
     * Each property should hold a path to the particular bundle format for the entry-point.
     * Defaults to all the properties in the package.json.
     */
    propertiesToConsider?: string[];
    /**
     * Whether to only process the typings files for this entry-point.
     *
     * This is useful when running ngcc only to provide typings files to downstream tooling such as
     * the Angular Language Service or ng-packagr. Defaults to `false`.
     *
     * If this is set to `true` then `compileAllFormats` is forced to `false`.
     */
    typingsOnly?: boolean;
    /**
     * Whether to process all formats specified by (`propertiesToConsider`)  or to stop processing
     * this entry-point at the first matching format.
     *
     * Defaults to `true`, but is forced to `false` if `typingsOnly` is `true`.
     */
    compileAllFormats?: boolean;
    /**
     * Whether to create new entry-points bundles rather than overwriting the original files.
     */
    createNewEntryPointFormats?: boolean;
    /**
     * Provide a logger that will be called with log messages.
     */
    logger?: Logger;
    /**
     * Paths mapping configuration (`paths` and `baseUrl`), as found in `ts.CompilerOptions`.
     * These are used to resolve paths to locally built Angular libraries.
     *
     * Note that `pathMappings` specified here take precedence over any `pathMappings` loaded from a
     * TS config file.
     */
    pathMappings?: PathMappings;
    /**
     * Provide a file-system service that will be used by ngcc for all file interactions.
     */
    fileSystem?: FileSystem;
    /**
     * Whether the compilation should run and return asynchronously. Allowing asynchronous execution
     * may speed up the compilation by utilizing multiple CPU cores (if available).
     *
     * Default: `false` (i.e. run synchronously)
     */
    async?: false;
    /**
     * Set to true in order to terminate immediately with an error code if an entry-point fails to be
     * processed.
     *
     * If `targetEntryPointPath` is provided then this property is always true and cannot be
     * changed. Otherwise the default is false.
     *
     * When set to false, ngcc will continue to process entry-points after a failure. In which case it
     * will log an error and resume processing other entry-points.
     */
    errorOnFailedEntryPoint?: boolean;
    /**
     * Render `$localize` messages with legacy format ids.
     *
     * The default value is `true`. Only set this to `false` if you do not want legacy message ids to
     * be rendered. For example, if you are not using legacy message ids in your translation files
     * AND are not doing compile-time inlining of translations, in which case the extra message ids
     * would add unwanted size to the final source bundle.
     *
     * It is safe to leave this set to true if you are doing compile-time inlining because the extra
     * legacy message ids will all be stripped during translation.
     */
    enableI18nLegacyMessageIdFormat?: boolean;
    /**
     * Whether to invalidate any entry-point manifest file that is on disk. Instead, walk the
     * directory tree looking for entry-points, and then write a new entry-point manifest, if
     * possible.
     *
     * Default: `false` (i.e. the manifest will be used if available)
     */
    invalidateEntryPointManifest?: boolean;
    /**
     * An absolute path to a TS config file (e.g. `tsconfig.json`) or a directory containing one, that
     * will be used to configure module resolution with things like path mappings, if not specified
     * explicitly via the `pathMappings` property to `mainNgcc`.
     *
     * If `undefined`, ngcc will attempt to load a `tsconfig.json` file from the directory above the
     * `basePath`.
     *
     * If `null`, ngcc will not attempt to load any TS config file at all.
     */
    tsConfigPath?: string | null;
    /**
     * Use the program defined in the loaded tsconfig.json (if available - see
     * `tsConfigPath` option) to identify the entry-points that should be processed.
     * If this is set to `true` then only the entry-points reachable from the given
     * program (and their dependencies) will be processed.
     */
    findEntryPointsFromTsConfigProgram?: boolean;
}
/**
 * The options to configure the ngcc compiler for asynchronous execution.
 */
export declare type AsyncNgccOptions = Omit<SyncNgccOptions, 'async'> & {
    async: true;
};
/**
 * The options to configure the ngcc compiler.
 */
export declare type NgccOptions = AsyncNgccOptions | SyncNgccOptions;
export declare type OptionalNgccOptionKeys = 'targetEntryPointPath' | 'tsConfigPath' | 'pathMappings' | 'findEntryPointsFromTsConfigProgram';
export declare type RequiredNgccOptions = Required<Omit<NgccOptions, OptionalNgccOptionKeys>>;
export declare type OptionalNgccOptions = Pick<NgccOptions, OptionalNgccOptionKeys>;
export declare type SharedSetup = {
    fileSystem: FileSystem;
    absBasePath: AbsoluteFsPath;
    projectPath: AbsoluteFsPath;
    tsConfig: ParsedConfiguration | null;
    getFileWriter(pkgJsonUpdater: PackageJsonUpdater): FileWriter;
};
/**
 * Instantiate common utilities that are always used and fix up options with defaults, as necessary.
 *
 * NOTE: Avoid eagerly instantiating anything that might not be used when running sync/async.
 */
export declare function getSharedSetup(options: NgccOptions): SharedSetup & RequiredNgccOptions & OptionalNgccOptions;
export declare function clearTsConfigCache(): void;
/**
 * Determines the maximum number of workers to use for parallel execution. This can be set using the
 * NGCC_MAX_WORKERS environment variable, or is computed based on the number of available CPUs. One
 * CPU core is always reserved for the master process, so we take the number of CPUs minus one, with
 * a maximum of 4 workers. We don't scale the number of workers beyond 4 by default, as it takes
 * considerably more memory and CPU cycles while not offering a substantial improvement in time.
 */
export declare function getMaxNumberOfWorkers(): number;
